
#include "320controller.h"
    
/* set_box - by Hill Robertson
 *         - Modified Greg Hormann - performance & Debugging & BSD args style
 * 
 * This function sends the 6 writes for each parallel port instruction
 * to CONTROL (currently) 40 8-circuit Christmas light SSR CONTROLLER
 * boxes.  The box and bank data are selected and then the write sequence
 * begins.
 *
 *
 * Note: If trying to use this on a LINUX system, be sure to reverse the
 *       arguments to outb()
 */

void set_box(int box, int data){
  
  int bank;          /* Which of the 5 banks:  8, 16, 32, 64, 128 */
  int bankbox;       /* Which box (0-7) in the bank. */
 
#ifdef DEBUG_BOX
  char temp[500];
#endif
  
#ifdef DEBUG_BOX    
  printf("Box %d ==> %d\n", box, data);
#endif
  
  if (box <=0 || box > 40)
  {
    printf("Invalid box %d\n", box);
    return;
  }

  /*
   * There are 4 banks of eight boxes. The banks 
   * are controlled by the 5 high bits (8,16,32,64,128)
   * while the boxes in each bank are binary coded (0-7)
   * by the first 3 pins.
   */

  if      (box<=8)   { bank=8; bankbox=box-1; }
  else if (box<=16)  { bank=16; bankbox=box-9; } 
  else if (box<=24)  { bank=32; bankbox=box-17; }
  else if (box<=32)  { bank=64; bankbox=box-25; }
  else               { bank=128; bankbox=box-33; }
  
  /*
   * BASE=0x378 for first parallel port BASE address
   * CONTROL=BASE+2 for first parallel port CONTROL address
   */

  /*
   * Write #1
   *   Outputs data byte (D0-D7) on pins 2-9
   *   of parallel port.  This is the data
   *   we are trying to send to box XX.
   */

  outb(BASE, data); 
  
#ifdef DEBUG_BOX    
  printf("S1: Base  = data (%d)\n", data);
  gets(temp);
#endif
  


  /*
   * Write #2:
   *   Outputs a 1 (high) on C0 and C1
   *   (pins 1 and 14) since they are inverted 
   *   without changing any states on the data pins.
   *
   *   This opens the "data buffer" flip-flop so that it
   *   can read the data on D0-D7.  It also opens up the 
   *   decoders for each bank solely to avoid the need for a 
   *   #7th write. 
   */
   
  outb(CONTROL, 0);       

#ifdef DEBUG_BOX    
  printf("S2: CONTROL  = 0 (C0/C1 ON) \n");
  gets(temp);
#endif
  


  /*
   * Write #3
   *   Outputs a 0 (low) on C0 and a 1 (high) on
   *   C1 since they are inverted. Again, not
   *   changing any states on the data pins
   *
   *   This "locks" the data presently on D0-D7
   *   into the data buffer (C0) but leaves the state
   *   of the decoders (C1) unchanged.
   */

  outb(CONTROL, 1);          // Write 3:
  
#ifdef DEBUG_BOX    
  printf("S3: CONTROL  = 1 (C0 (low) C1 (high) \n");
  gets(temp);
#endif


  /*
   * Write #4
   *    Outputs the steering (addressing) data on
   *    the data pins
   *
   *    The "bankbox" says which bank should be listening 
   *    (8, 16, 32, 64, or 127) while the low 3 bits say
   *    which of the eight flip-flops (0-7) in that bank 
   *    should be listening.
   */

  outb(BASE, bankbox+bank);  // Write 4:
  
#ifdef DEBUG_BOX    
  printf("S4: BASE = bankbox+bank (%d)\n", bankbox+bank);
  gets(temp);
#endif


  /*
   * Write #5
   *   Outputs a 0 (low) on both C0 and C1
   *   since they are inverted
   *
   *  This locks the data into the correct 
   *  decoder which sends a "low" single 
   *  to the clock port of one of the 40 flip flops.
   *  allowing the data from the "buffer" flip flop
   *  to flow in
   */

  outb(CONTROL, 3);          // Write 5:
  
#ifdef DEBUG_BOX    
  printf("S5: CONTROL = 3 (LOW C0, C1)\n");
  gets(temp);
#endif


  /*
   * Write #6
   *   Outputs a 0 (low) on C0 and a 1 (high) on
   *   C1 since they are inverted. Again, not
   *   changing any states on the data pins
   *
   *   This locks the data into the correct
   *   flip flop and will remain transmitting this data
   *   to the relays untill the next time this
   *   box needs to be modified.
   */
  
  outb(CONTROL, 1);          // Write 6:
  
#ifdef DEBUG_BOX    
  printf("S6: CONTROL = 1 (LOW C0, HIGH C1)\n") ;
  gets(temp);
#endif

} /* Send Sent_box function


/*  LocalWords:  SSR outb int bankbox  ifdef endif printf
 */
